#pragma once
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <initializer_list>
#include <iterator>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 201703L
#include <string_view>
#endif
#if __cplusplus >= 202002L
#include <ranges>
#endif

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;
#if __cplusplus >= 201703L
  explicit operator std::string_view() const;
#endif

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BOX
#define CXXBRIDGE1_RUST_BOX
template <typename T>
class Box final {
public:
  using element_type = T;
  using const_pointer =
      typename std::add_pointer<typename std::add_const<T>::type>::type;
  using pointer = typename std::add_pointer<T>::type;

  Box() = delete;
  Box(Box &&) noexcept;
  ~Box() noexcept;

  explicit Box(const T &);
  explicit Box(T &&);

  Box &operator=(Box &&) & noexcept;

  const T *operator->() const noexcept;
  const T &operator*() const noexcept;
  T *operator->() noexcept;
  T &operator*() noexcept;

  template <typename... Fields>
  static Box in_place(Fields &&...);

  void swap(Box &) noexcept;

  static Box from_raw(T *) noexcept;

  T *into_raw() noexcept;

  /* Deprecated */ using value_type = element_type;

private:
  class uninit;
  class allocation;
  Box(uninit) noexcept;
  void drop() noexcept;

  friend void swap(Box &lhs, Box &rhs) noexcept { lhs.swap(rhs); }

  T *ptr;
};

template <typename T>
class Box<T>::uninit {};

template <typename T>
class Box<T>::allocation {
  static T *alloc() noexcept;
  static void dealloc(T *) noexcept;

public:
  allocation() noexcept : ptr(alloc()) {}
  ~allocation() noexcept {
    if (this->ptr) {
      dealloc(this->ptr);
    }
  }
  T *ptr;
};

template <typename T>
Box<T>::Box(Box &&other) noexcept : ptr(other.ptr) {
  other.ptr = nullptr;
}

template <typename T>
Box<T>::Box(const T &val) {
  allocation alloc;
  ::new (alloc.ptr) T(val);
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::Box(T &&val) {
  allocation alloc;
  ::new (alloc.ptr) T(std::move(val));
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::~Box() noexcept {
  if (this->ptr) {
    this->drop();
  }
}

template <typename T>
Box<T> &Box<T>::operator=(Box &&other) & noexcept {
  if (this->ptr) {
    this->drop();
  }
  this->ptr = other.ptr;
  other.ptr = nullptr;
  return *this;
}

template <typename T>
const T *Box<T>::operator->() const noexcept {
  return this->ptr;
}

template <typename T>
const T &Box<T>::operator*() const noexcept {
  return *this->ptr;
}

template <typename T>
T *Box<T>::operator->() noexcept {
  return this->ptr;
}

template <typename T>
T &Box<T>::operator*() noexcept {
  return *this->ptr;
}

template <typename T>
template <typename... Fields>
Box<T> Box<T>::in_place(Fields &&...fields) {
  allocation alloc;
  auto ptr = alloc.ptr;
  ::new (ptr) T{std::forward<Fields>(fields)...};
  alloc.ptr = nullptr;
  return from_raw(ptr);
}

template <typename T>
void Box<T>::swap(Box &rhs) noexcept {
  using std::swap;
  swap(this->ptr, rhs.ptr);
}

template <typename T>
Box<T> Box<T>::from_raw(T *raw) noexcept {
  Box box = uninit{};
  box.ptr = raw;
  return box;
}

template <typename T>
T *Box<T>::into_raw() noexcept {
  T *raw = this->ptr;
  this->ptr = nullptr;
  return raw;
}

template <typename T>
Box<T>::Box(uninit) noexcept {}
#endif // CXXBRIDGE1_RUST_BOX

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_OPAQUE
#define CXXBRIDGE1_RUST_OPAQUE
class Opaque {
public:
  Opaque() = delete;
  Opaque(const Opaque &) = delete;
  ~Opaque() = delete;
};
#endif // CXXBRIDGE1_RUST_OPAQUE

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT
} // namespace cxxbridge1
} // namespace rust

#if __cplusplus >= 201402L
#define CXX_DEFAULT_VALUE(value) = value
#else
#define CXX_DEFAULT_VALUE(value)
#endif

namespace loot {
  namespace rust {
    enum class GameType : ::std::uint8_t;
    enum class MessageType : ::std::uint8_t;
    enum class TagSuggestion : ::std::uint8_t;
    enum class EdgeType : ::std::uint8_t;
    enum class LogLevel : ::std::uint8_t;
    struct OptionalMessageContentRef;
    struct Game;
    struct Database;
    struct Message;
    struct MessageContent;
    struct Group;
    struct Plugin;
    struct OptionalPlugin;
    struct OptionalCrc;
    struct Vertex;
    struct OptionalPluginMetadata;
    struct PluginMetadata;
    struct File;
    struct Filename;
    struct Tag;
    struct PluginCleaningData;
    struct Location;
  }
}

namespace loot {
namespace rust {
#ifndef CXXBRIDGE1_ENUM_loot$rust$GameType
#define CXXBRIDGE1_ENUM_loot$rust$GameType
enum class GameType : ::std::uint8_t {
  Oblivion = 0,
  Skyrim = 1,
  Fallout3 = 2,
  FalloutNV = 3,
  Fallout4 = 4,
  SkyrimSE = 5,
  Fallout4VR = 6,
  SkyrimVR = 7,
  Morrowind = 8,
  Starfield = 9,
  OpenMW = 10,
  OblivionRemastered = 11,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$GameType

#ifndef CXXBRIDGE1_ENUM_loot$rust$MessageType
#define CXXBRIDGE1_ENUM_loot$rust$MessageType
enum class MessageType : ::std::uint8_t {
  Say = 0,
  Warn = 1,
  Error = 2,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$MessageType

#ifndef CXXBRIDGE1_ENUM_loot$rust$TagSuggestion
#define CXXBRIDGE1_ENUM_loot$rust$TagSuggestion
enum class TagSuggestion : ::std::uint8_t {
  Addition = 0,
  Removal = 1,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$TagSuggestion

#ifndef CXXBRIDGE1_ENUM_loot$rust$EdgeType
#define CXXBRIDGE1_ENUM_loot$rust$EdgeType
enum class EdgeType : ::std::uint8_t {
  // A special value that indicates that there is no edge.
  None = 0,
  Hardcoded = 1,
  MasterFlag = 2,
  Master = 3,
  MasterlistRequirement = 4,
  UserRequirement = 5,
  MasterlistLoadAfter = 6,
  UserLoadAfter = 7,
  MasterlistGroup = 8,
  UserGroup = 9,
  RecordOverlap = 10,
  AssetOverlap = 11,
  TieBreak = 12,
  BlueprintMaster = 13,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$EdgeType

#ifndef CXXBRIDGE1_ENUM_loot$rust$LogLevel
#define CXXBRIDGE1_ENUM_loot$rust$LogLevel
enum class LogLevel : ::std::uint8_t {
  Trace = 0,
  Debug = 1,
  Info = 2,
  Warning = 3,
  Error = 4,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$LogLevel

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalMessageContentRef
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalMessageContentRef
struct OptionalMessageContentRef final {
  ::loot::rust::MessageContent const *pointer CXX_DEFAULT_VALUE(nullptr);

  bool is_some() const noexcept;
  ::loot::rust::MessageContent const &as_ref() const;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalMessageContentRef

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Game
#define CXXBRIDGE1_STRUCT_loot$rust$Game
struct Game final : public ::rust::Opaque {
  ::loot::rust::GameType game_type() const;
  ::rust::Vec<::rust::String> additional_data_paths() const;
  void set_additional_data_paths(::rust::Slice<::rust::Str const> additional_data_paths);
  ::rust::Box<::loot::rust::Database> database() const noexcept;
  bool is_valid_plugin(::rust::Str plugin_path) const noexcept;
  void load_plugins(::rust::Slice<::rust::Str const> plugin_paths);
  void load_plugin_headers(::rust::Slice<::rust::Str const> plugin_paths);
  void clear_loaded_plugins() noexcept;
  ::rust::Box<::loot::rust::OptionalPlugin> plugin(::rust::Str plugin_name) const noexcept;
  ::rust::Vec<::loot::rust::Plugin> loaded_plugins() const noexcept;
  ::rust::Vec<::rust::String> sort_plugins(::rust::Slice<::rust::Str const> plugin_names) const;
  void load_current_load_order_state();
  bool is_load_order_ambiguous() const;
  ::rust::String active_plugins_file_path() const;
  bool is_plugin_active(::rust::Str plugin_name) const noexcept;
  ::rust::Vec<::rust::String> load_order() const noexcept;
  void set_load_order(::rust::Slice<::rust::Str const> load_order);
  ~Game() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Game

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Database
#define CXXBRIDGE1_STRUCT_loot$rust$Database
struct Database final : public ::rust::Opaque {
  void load_masterlist(::rust::Str path) const;
  void load_masterlist_with_prelude(::rust::Str masterlist_path, ::rust::Str prelude_path) const;
  void load_userlist(::rust::Str path) const;
  void write_user_metadata(::rust::Str output_path, bool overwrite) const;
  void write_minimal_list(::rust::Str output_path, bool overwrite) const;
  bool evaluate(::rust::Str condition) const;
  ::rust::Vec<::rust::String> known_bash_tags() const;
  ::rust::Vec<::loot::rust::Message> general_messages(bool evaluate_conditions) const;
  ::rust::Vec<::loot::rust::Group> groups(bool include_user_metadata) const;
  ::rust::Vec<::loot::rust::Group> user_groups() const;
  void set_user_groups(::rust::Slice<::rust::Box<::loot::rust::Group> const> groups) const;
  ::rust::Vec<::loot::rust::Vertex> groups_path(::rust::Str from_group_name, ::rust::Str to_group_name) const;
  ::rust::Box<::loot::rust::OptionalPluginMetadata> plugin_metadata(::rust::Str plugin_name, bool include_user_metadata, bool evaluate_conditions) const;
  ::rust::Box<::loot::rust::OptionalPluginMetadata> plugin_user_metadata(::rust::Str plugin_name, bool evaluate_conditions) const;
  void set_plugin_user_metadata(::rust::Box<::loot::rust::PluginMetadata> plugin_metadata);
  void discard_plugin_user_metadata(::rust::Str plugin) const;
  void discard_all_user_metadata() const;
  ~Database() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Database

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Message
#define CXXBRIDGE1_STRUCT_loot$rust$Message
struct Message final : public ::rust::Opaque {
  ::loot::rust::MessageType message_type() const noexcept;
  ::rust::Slice<::loot::rust::MessageContent const> content() const noexcept;
  ::rust::Str condition() const noexcept;
  ::rust::Box<::loot::rust::Message> boxed_clone() const noexcept;
  ~Message() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Message

#ifndef CXXBRIDGE1_STRUCT_loot$rust$MessageContent
#define CXXBRIDGE1_STRUCT_loot$rust$MessageContent
struct MessageContent final : public ::rust::Opaque {
  ::rust::Str text() const noexcept;
  ::rust::Str language() const noexcept;
  ::rust::Box<::loot::rust::MessageContent> boxed_clone() const noexcept;
  ~MessageContent() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$MessageContent

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Group
#define CXXBRIDGE1_STRUCT_loot$rust$Group
struct Group final : public ::rust::Opaque {
  ::rust::Box<::loot::rust::Group> boxed_clone() const noexcept;
  ::rust::Str name() const noexcept;
  ::rust::Str description() const noexcept;
  ::rust::Slice<::rust::String const> after_groups() const noexcept;
  ~Group() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Group

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Plugin
#define CXXBRIDGE1_STRUCT_loot$rust$Plugin
struct Plugin final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;

  // NaN is used to indicate that the header version was not found.
  float header_version() const noexcept;

  // An empty string is used to indicate that no version was found.
  ::rust::Str version() const noexcept;

  ::rust::Vec<::rust::String> masters() const;
  ::rust::Slice<::rust::String const> bash_tags() const noexcept;
  ::rust::Box<::loot::rust::OptionalCrc> crc() const noexcept;
  bool is_master() const noexcept;
  bool is_light_plugin() const noexcept;
  bool is_medium_plugin() const noexcept;
  bool is_update_plugin() const noexcept;
  bool is_blueprint_plugin() const noexcept;
  bool is_valid_as_light_plugin() const;
  bool is_valid_as_medium_plugin() const;
  bool is_valid_as_update_plugin() const;
  bool is_empty() const noexcept;
  bool loads_archive() const noexcept;
  bool do_records_overlap(::loot::rust::Plugin const &plugin) const;
  ::rust::Box<::loot::rust::Plugin> boxed_clone() const noexcept;
  ~Plugin() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Plugin

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalPlugin
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalPlugin
struct OptionalPlugin final : public ::rust::Opaque {
  bool is_some() const noexcept;
  ::loot::rust::Plugin const &as_ref() const;
  ~OptionalPlugin() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalPlugin

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalCrc
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalCrc
struct OptionalCrc final : public ::rust::Opaque {
  bool is_some() const noexcept;
  ::std::uint32_t const &as_ref() const;
  ~OptionalCrc() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalCrc

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Vertex
#define CXXBRIDGE1_STRUCT_loot$rust$Vertex
struct Vertex final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;
  ::loot::rust::EdgeType out_edge_type() const;
  ::rust::Box<::loot::rust::Vertex> boxed_clone() const noexcept;
  ~Vertex() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Vertex

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalPluginMetadata
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalPluginMetadata
struct OptionalPluginMetadata final : public ::rust::Opaque {
  bool is_some() const noexcept;
  ::loot::rust::PluginMetadata const &as_ref() const;
  ~OptionalPluginMetadata() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalPluginMetadata

#ifndef CXXBRIDGE1_STRUCT_loot$rust$PluginMetadata
#define CXXBRIDGE1_STRUCT_loot$rust$PluginMetadata
struct PluginMetadata final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;

  // An empty string is used to indicate that no group is set.
  ::rust::Str group() const noexcept;

  ::rust::Slice<::loot::rust::File const> load_after_files() const noexcept;
  ::rust::Slice<::loot::rust::File const> requirements() const noexcept;
  ::rust::Slice<::loot::rust::File const> incompatibilities() const noexcept;
  ::rust::Slice<::loot::rust::Message const> messages() const noexcept;
  ::rust::Slice<::loot::rust::Tag const> tags() const noexcept;
  ::rust::Slice<::loot::rust::PluginCleaningData const> dirty_info() const noexcept;
  ::rust::Slice<::loot::rust::PluginCleaningData const> clean_info() const noexcept;
  ::rust::Slice<::loot::rust::Location const> locations() const noexcept;
  void set_group(::rust::String group) noexcept;
  void unset_group() noexcept;
  void set_load_after_files(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;
  void set_requirements(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;
  void set_incompatibilities(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;
  void set_messages(::rust::Slice<::rust::Box<::loot::rust::Message> const> messages) noexcept;
  void set_tags(::rust::Slice<::rust::Box<::loot::rust::Tag> const> tags) noexcept;
  void set_dirty_info(::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept;
  void set_clean_info(::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept;
  void set_locations(::rust::Slice<::rust::Box<::loot::rust::Location> const> locations) noexcept;
  void merge_metadata(::loot::rust::PluginMetadata const &plugin) noexcept;
  bool has_name_only() const noexcept;
  bool is_regex_plugin() const noexcept;
  bool name_matches(::rust::Str other_name) const noexcept;
  ::rust::String as_yaml() const noexcept;
  ::rust::Box<::loot::rust::PluginMetadata> boxed_clone() const noexcept;
  ~PluginMetadata() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$PluginMetadata

#ifndef CXXBRIDGE1_STRUCT_loot$rust$File
#define CXXBRIDGE1_STRUCT_loot$rust$File
struct File final : public ::rust::Opaque {
  ::loot::rust::Filename const &filename() const noexcept;
  ::rust::Str display_name() const noexcept;
  ::rust::Slice<::loot::rust::MessageContent const> detail() const noexcept;
  ::rust::Str condition() const noexcept;
  ::rust::Str constraint() const noexcept;
  ::rust::Box<::loot::rust::File> boxed_clone() const noexcept;
  ~File() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$File

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Filename
#define CXXBRIDGE1_STRUCT_loot$rust$Filename
struct Filename final : public ::rust::Opaque {
  ::rust::Str as_str() const noexcept;
  ::rust::Box<::loot::rust::Filename> boxed_clone() const noexcept;
  bool eq(::loot::rust::Filename const &other) const noexcept;
  bool ne(::loot::rust::Filename const &other) const noexcept;
  bool lt(::loot::rust::Filename const &other) const noexcept;
  bool le(::loot::rust::Filename const &other) const noexcept;
  bool gt(::loot::rust::Filename const &other) const noexcept;
  bool ge(::loot::rust::Filename const &other) const noexcept;
  ~Filename() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Filename

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Tag
#define CXXBRIDGE1_STRUCT_loot$rust$Tag
struct Tag final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;
  bool is_addition() const noexcept;
  ::rust::Str condition() const noexcept;
  ::rust::Box<::loot::rust::Tag> boxed_clone() const noexcept;
  ~Tag() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Tag

#ifndef CXXBRIDGE1_STRUCT_loot$rust$PluginCleaningData
#define CXXBRIDGE1_STRUCT_loot$rust$PluginCleaningData
struct PluginCleaningData final : public ::rust::Opaque {
  ::std::uint32_t crc() const noexcept;
  ::std::uint32_t itm_count() const noexcept;
  ::std::uint32_t deleted_reference_count() const noexcept;
  ::std::uint32_t deleted_navmesh_count() const noexcept;
  ::rust::Str cleaning_utility() const noexcept;
  ::rust::Slice<::loot::rust::MessageContent const> detail() const noexcept;
  ::rust::Box<::loot::rust::PluginCleaningData> boxed_clone() const noexcept;
  ~PluginCleaningData() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$PluginCleaningData

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Location
#define CXXBRIDGE1_STRUCT_loot$rust$Location
struct Location final : public ::rust::Opaque {
  ::rust::Str url() const noexcept;
  ::rust::Str name() const noexcept;
  ::rust::Box<::loot::rust::Location> boxed_clone() const noexcept;
  ~Location() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Location

void set_log_level(::loot::rust::LogLevel level);

bool is_compatible(::std::uint32_t major, ::std::uint32_t minor, ::std::uint32_t patch) noexcept;

::rust::String libloot_version() noexcept;

::rust::String libloot_revision() noexcept;

::loot::rust::OptionalMessageContentRef select_message_content(::rust::Slice<::loot::rust::MessageContent const> contents, ::rust::Str language) noexcept;

::rust::Box<::loot::rust::Game> new_game(::loot::rust::GameType game_type, ::rust::Str game_path);

::rust::Box<::loot::rust::Game> new_game_with_local_path(::loot::rust::GameType game_type, ::rust::Str game_path, ::rust::Str game_local_path);

::rust::Box<::loot::rust::Message> new_message(::loot::rust::MessageType message_type, ::rust::String content, ::rust::Str condition);

::rust::Box<::loot::rust::Message> multilingual_message(::loot::rust::MessageType message_type, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> contents, ::rust::Str condition);

::rust::Str message_content_default_language() noexcept;

::rust::Box<::loot::rust::MessageContent> new_message_content(::rust::String text, ::rust::Str language) noexcept;

::rust::Box<::loot::rust::Group> new_group(::rust::String name, ::rust::Str description, ::rust::Vec<::rust::String> after_groups) noexcept;

::rust::Str group_default_name() noexcept;

::rust::Box<::loot::rust::Vertex> new_vertex(::rust::String name, ::loot::rust::EdgeType out_edge_type);

::rust::Box<::loot::rust::PluginMetadata> new_plugin_metadata(::rust::Str name);

::rust::Box<::loot::rust::File> new_file(::rust::String name, ::rust::Str display_name, ::rust::Str condition, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> detail, ::rust::Str constraint);

::rust::Box<::loot::rust::Filename> new_filename(::rust::String name) noexcept;

::rust::Box<::loot::rust::Tag> new_tag(::rust::String name, ::loot::rust::TagSuggestion suggestion, ::rust::Str condition);

::rust::Box<::loot::rust::PluginCleaningData> new_plugin_cleaning_data(::std::uint32_t crc, ::rust::String cleaning_utility, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> detail, ::std::uint32_t itm_count, ::std::uint32_t deleted_reference_count, ::std::uint32_t deleted_navmesh_count);

::rust::Box<::loot::rust::Location> new_location(::rust::String url, ::rust::Str name) noexcept;
} // namespace rust
} // namespace loot
